// Photo Courses Website - Version 4 (HTML/CSS/JS)

const courseLevels = [
  { 
    id: 'beginner', 
    name: 'Начинающий', 
    icon: 'icons/level-beginner.svg',
    color: '#2563EB',
    description: 'Основы фотографии для новичков',
    duration: '8 недель',
    lessons: '24 урока'
  },
  { 
    id: 'advanced', 
    name: 'Продвинутый', 
    icon: 'icons/level-advanced.svg',
    color: '#3B82F6',
    description: 'Углубленное изучение техник',
    duration: '12 недель',
    lessons: '36 уроков'
  },
  { 
    id: 'professional', 
    name: 'Профессиональный', 
    icon: 'icons/level-pro.svg',
    color: '#60A5FA',
    description: 'Мастерство для профессионалов',
    duration: '16 недель',
    lessons: '48 уроков'
  },
  { 
    id: 'retouch', 
    name: 'Ретушь', 
    icon: 'icons/level-retouch.svg',
    color: '#2563EB',
    description: 'Обработка и ретушь фотографий',
    duration: '6 недель',
    lessons: '18 уроков'
  }
]

const instructors = [
  { name: 'Анна Петрова', experience: '12 лет', rating: 5, specialty: 'Портретная съемка', students: '5000+' },
  { name: 'Михаил Соколов', experience: '15 лет', rating: 5, specialty: 'Пейзажная фотография', students: '7000+' },
  { name: 'Елена Волкова', experience: '10 лет', rating: 5, specialty: 'Студийная съемка', students: '4000+' },
  { name: 'Дмитрий Иванов', experience: '18 лет', rating: 5, specialty: 'Ретушь и обработка', students: '9000+' }
]

const guarantees = [
  { icon: 'icons/guarantee-shield.svg', title: 'Гарантия качества', text: 'Возврат средств при неудовлетворенности' },
  { icon: 'icons/guarantee-clock.svg', title: 'Гибкий график', text: 'Учитесь в удобное время' },
  { icon: 'icons/guarantee-users.svg', title: 'Поддержка', text: 'Помощь на всех этапах обучения' },
  { icon: 'icons/guarantee-award.svg', title: 'Сертификат', text: 'Официальный документ по окончании' }
]

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initPage() {
  const currentPage = window.location.pathname.includes('enroll.html') ? 'enroll' : 'index'
  
  if (currentPage === 'enroll') {
    initEnrollPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadCourses()
  updateStats()
  
  setInterval(() => {
    loadCourses()
    updateStats()
  }, 500)
}

function loadCourses() {
  const saved = localStorage.getItem('photoCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  } else {
    courses = [
      {
        id: 1,
        module: 'Основы композиции',
        instructor: 'Анна Петрова',
        completionDate: '2025-03-25',
        rating: 5
      },
      {
        id: 2,
        module: 'Работа со светом',
        instructor: 'Михаил Соколов',
        completionDate: '2025-03-20',
        rating: 5
      },
      {
        id: 3,
        module: 'Портретная съемка',
        instructor: 'Елена Волкова',
        completionDate: '2025-03-15',
        rating: 4
      }
    ]
    localStorage.setItem('photoCourses', JSON.stringify(courses))
  }
  
  const coursesList = document.getElementById('coursesList')
  if (coursesList) {
    if (courses.length === 0) {
      coursesList.innerHTML = `
        <div class="empty-state">
          <img src="icons/logo-camera.svg" alt="Camera" width="64" height="64">
          <h3 class="empty-title">Нет курсов</h3>
          <p class="empty-text">Запишитесь на курс прямо сейчас</p>
        </div>
      `
    } else {
      coursesList.innerHTML = courses.map(course => {
        const stars = Array.from({ length: 5 }, (_, i) => 
          `<span class="star">${i < course.rating ? '★' : '☆'}</span>`
        ).join('')
        
        const date = new Date(course.completionDate).toLocaleDateString('ru-RU', { 
          day: 'numeric', 
          month: 'long', 
          year: 'numeric' 
        })
        
        return `
          <div class="course-item">
            <div class="course-main">
              <div class="course-info">
                <h3 class="course-module">${course.module}</h3>
                <div class="course-details">
                  <div class="course-detail">
                    <img src="icons/course-instructor.svg" alt="Instructor" width="16" height="16">
                    <span>${course.instructor}</span>
                  </div>
                  <div class="course-detail">
                    <img src="icons/course-date.svg" alt="Date" width="16" height="16">
                    <span>${date}</span>
                  </div>
                </div>
              </div>
              <div class="course-rating">${stars}</div>
            </div>
          </div>
        `
      }).join('')
    }
  }
}

function updateStats() {
  const saved = localStorage.getItem('photoCourses')
  let courses = []
  
  if (saved) {
    try {
      courses = JSON.parse(saved)
    } catch (e) {
      courses = []
    }
  }
  
  const totalCourses = courses.length
  const uniqueInstructors = new Set(courses.map(c => c.instructor)).size
  const averageRating = courses.length > 0 
    ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1)
    : 0
  
  const stats = [
    {
      value: totalCourses,
      label: 'Курсов',
      icon: 'icons/stat-courses.svg'
    },
    {
      value: averageRating,
      label: 'Средняя оценка',
      icon: 'icons/stat-rating.svg'
    },
    {
      value: uniqueInstructors,
      label: 'Преподавателей',
      icon: 'icons/stat-instructors.svg'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <img src="${stat.icon}" alt="" width="32" height="32">
        <div class="stat-content">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
}

function initEnrollPage() {
  const enrollmentForm = document.getElementById('enrollmentForm')
  const levelsList = document.getElementById('levelsList')
  const dateInput = document.getElementById('startDate')
  const successMessage = document.getElementById('successMessage')
  const guaranteesGrid = document.getElementById('guaranteesGrid')
  const instructorsGrid = document.getElementById('instructorsGrid')

  let selectedLevel = null

  if (levelsList) {
    levelsList.innerHTML = courseLevels.map(level => `
      <button type="button" class="level-item" data-id="${level.id}">
        <div class="level-icon">
          <img src="${level.icon}" alt="${level.name}" width="28" height="28">
        </div>
        <div class="level-content">
          <h3 class="level-name">${level.name}</h3>
          <p class="level-description">${level.description}</p>
          <div class="level-meta">
            <span class="level-duration">
              <img src="icons/guarantee-clock.svg" alt="Clock" width="16" height="16">
              ${level.duration}
            </span>
            <span class="level-lessons">
              <img src="icons/guarantee-award.svg" alt="Book" width="16" height="16">
              ${level.lessons}
            </span>
          </div>
        </div>
      </button>
    `).join('')

    levelsList.querySelectorAll('.level-item').forEach(button => {
      button.addEventListener('click', function() {
        levelsList.querySelectorAll('.level-item').forEach(btn => {
          btn.classList.remove('selected')
          const check = btn.querySelector('.level-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        selectedLevel = courseLevels.find(l => l.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'level-check'
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="24" height="24">`
        this.appendChild(check)
      })
    })
  }

  if (guaranteesGrid) {
    guaranteesGrid.innerHTML = guarantees.map(guarantee => `
      <div class="guarantee-item">
        <div class="guarantee-icon">
          <img src="${guarantee.icon}" alt="" width="28" height="28">
        </div>
        <h3 class="guarantee-title">${guarantee.title}</h3>
        <p class="guarantee-text">${guarantee.text}</p>
      </div>
    `).join('')
  }

  if (instructorsGrid) {
    instructorsGrid.innerHTML = instructors.map(instructor => {
      const stars = Array.from({ length: 5 }, () => '★').join('')
      
      return `
        <div class="instructor-card">
          <div class="instructor-avatar">
            <img src="icons/instructor-avatar.svg" alt="" width="32" height="32">
          </div>
          <h3 class="instructor-name">${instructor.name}</h3>
          <p class="instructor-experience">${instructor.experience} опыта</p>
          <div class="instructor-rating">
            <span class="star">${stars}</span>
          </div>
          <p class="instructor-specialty">${instructor.specialty}</p>
          <div class="instructor-students">
            <img src="icons/guarantee-users.svg" alt="" width="14" height="14">
            <span>${instructor.students} студентов</span>
          </div>
        </div>
      `
    }).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (enrollmentForm) {
    enrollmentForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedLevel && dateInput.value) {
        const savedCourses = localStorage.getItem('photoCourses')
        const courses = savedCourses ? JSON.parse(savedCourses) : []
        
        const randomInstructor = instructors[Math.floor(Math.random() * instructors.length)]
        const weeks = parseInt(selectedLevel.duration)
        const completionDate = new Date(new Date(dateInput.value).getTime() + weeks * 7 * 24 * 60 * 60 * 1000)
        
        const newCourse = {
          id: Date.now(),
          module: selectedLevel.name,
          instructor: randomInstructor.name,
          completionDate: completionDate.toISOString().split('T')[0],
          rating: 0
        }

        courses.push(newCourse)
        localStorage.setItem('photoCourses', JSON.stringify(courses))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedLevel = null
          dateInput.value = ''
          levelsList.querySelectorAll('.level-item').forEach(btn => {
            btn.classList.remove('selected')
            const check = btn.querySelector('.level-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  if (slides.length === 0) return
  
  slides[currentSlide].classList.remove('active')
  if (dots[currentSlide]) dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  if (dots[currentSlide]) dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

